/** @file   particlesystem.h
 * @brief   Declaration of ParticleSystem - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_PARTICLESYSTEM_H
#define H_ENG2D_PARTICLESYSTEM_H

#include "eng2d_dll.h"
#include "interactive.h"
#include <vector>
#include "particle.h"


namespace eng2d {


/** @class  ParticleSystem
 * @brief   This manages the set of Particles.
 * @author  Tomi Lamminsaari
 *
 * Every Particle belongs to a ParticleSystem. This class manages and updates
 * all the particles added to it.
 *
 * Usually this class operates as a parent class for the actual particle
 * effects such as ParticleBlood or ParticleGlass.
 */
class DLLIMPORT ParticleSystem : public Interactive
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	ParticleSystem();


	/** Destructor
   */
	virtual ~ParticleSystem();

private:

	/** Copy constructor.
   * @param     rO                Reference to another ParticleSystem
   */
  ParticleSystem( const ParticleSystem& rO );

	/** Assignment operator
   * @param     rO                Reference to another ParticleSystem
   * @return    Reference to us.
   */
  ParticleSystem& operator = ( const ParticleSystem& rO );

public:


  ///
  /// Methods
  /// =======
  
  /** Adds the given particle to this particle system. The ownership of the
   * given particle is given to this instance of ParticleSystem.
   * @param     pP                Pointer to Particle being added.
   */
  virtual void addParticle( Particle* pP );

  /** This method updates all the particles.
   * @return    Number of active particles there are still left.
   */
  virtual void update();
  
  /** Redraws this ParticleSystem on given bitmap.
   * @param     pB                Pointer to target bitmap
   * @param     offset            Offset to translate from world-coordinates
   *                              to screen coordinates.
   */
  virtual void redraw( BITMAP* pB, const Vec2D& offset ) const;
  


  ///
  /// Getter methods
  /// ==============

  /** Tells if all the particles in this system has died. This indicates that
   * this instance of ParticleSystem can be destroyed. Before this method
   * works, you must call the @c update() - method first.
   * @return    <code>true</code> if all the particles have died.
   */
  bool dead() const;
  


  ///
  /// Members
  /// =======

  /** The particles are stored here. */
  std::vector< Particle* >  m_particles;

  /** Number of active particles there still are. This gets valid value
   * after you've called @c update() - method.
   */
  int m_activeParticleCount;

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace


#endif
